import { defineStore } from 'pinia';
import type { ApiResponse } from '~/types/api';

interface Transaction {
  id: number;
  course_id: number;
  course_title: string;
  enrollment_id: number;
  amount: string;
  payment_method: string;
  transaction_id: string;
  payment_status: string;
  created_at: string;
}

interface TransactionState {
  transactions: Transaction[];
  loading: boolean;
  error: string | null;
  pagination: {
    current_page: number;
    total_pages: number;
    per_page: number;
    total: number;
  } | null;
}

export const useTransactionsStore = defineStore('transactions', {
  state: (): TransactionState => ({
    transactions: [],
    loading: false,
    error: null,
    pagination: null,
  }),

  getters: {
    /**
     * Get successful transactions
     */
    successfulTransactions: (state) => 
      state.transactions.filter(t => t.payment_status === 'completed' || t.payment_status === 'success'),

    /**
     * Get pending transactions
     */
    pendingTransactions: (state) => 
      state.transactions.filter(t => t.payment_status === 'pending'),

    /**
     * Get failed transactions
     */
    failedTransactions: (state) => 
      state.transactions.filter(t => t.payment_status === 'failed' || t.payment_status === 'cancelled'),

    /**
     * Calculate total spent
     */
    totalSpent: (state) => 
      state.transactions
        .filter(t => t.payment_status === 'completed' || t.payment_status === 'success')
        .reduce((sum, t) => sum + parseFloat(t.amount || '0'), 0),
  },

  actions: {
    /**
     * Fetch transaction history
     */
    async fetchTransactions(page: number = 1, perPage: number = 10) {
      this.loading = true;
      this.error = null;

      try {
        const config = useRuntimeConfig();
        const token = useCookie('access_token');

        const response = await $fetch<ApiResponse<{
        transactions: Transaction[];
        current_page: number;
        total_pages: number;
        per_page: number;
        total: number;
      }>>(`${config.public.apiBase}/student/transactions`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${token.value}`,
          },
          params: {
            page,
            per_page: perPage,
          },
        });

        if (response.success) {
          this.transactions = response.data.transactions || [];
          this.pagination = {
            current_page: response.data.current_page || 1,
            total_pages: response.data.total_pages || 1,
            per_page: response.data.per_page || 10,
            total: response.data.total || 0,
          };
        } else {
          throw new Error(response.message || 'Failed to fetch transactions');
        }
      } catch (error: any) {
        console.error('Fetch transactions error:', error);
        this.error = error.message || 'Failed to fetch transactions';
        throw error;
      } finally {
        this.loading = false;
      }
    },

    /**
     * Get transaction by ID
     */
    async fetchTransactionById(transactionId: number) {
      try {
        const config = useRuntimeConfig();
        const token = useCookie('access_token');

        const response = await $fetch<ApiResponse<Transaction>>(`${config.public.apiBase}/student/transactions/${transactionId}`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${token.value}`,
          },
        });

        if (response.success) {
          return response.data;
        } else {
          throw new Error(response.message || 'Failed to fetch transaction');
        }
      } catch (error: any) {
        console.error('Fetch transaction error:', error);
        throw error;
      }
    },

    /**
     * Reset transactions state
     */
    reset() {
      this.transactions = [];
      this.loading = false;
      this.error = null;
      this.pagination = null;
    },
  },
});
