import { defineStore } from 'pinia';
import type { ApiResponse } from '~/types/api';

interface Category {
  id: number;
  name: string;
  slug: string;
  courses_count?: number;
}

interface SearchState {
  query: string;
  results: any[];
  categories: Category[];
  loading: boolean;
  error: string | null;
  categoriesFetched: boolean; // Add flag to prevent duplicate fetches
  pagination: {
    currentPage: number;
    totalPages: number;
    total: number;
    perPage: number;
  };
}

export const useSearchStore = defineStore('search', {
  state: (): SearchState => ({
    query: '',
    results: [],
    categories: [],
    loading: false,
    error: null,
    categoriesFetched: false,
    pagination: {
      currentPage: 1,
      totalPages: 1,
      total: 0,
      perPage: 12,
    },
  }),

  actions: {
    /**
     * Search courses with filters
     */
    async searchCourses(params: any = {}) {
      this.loading = true;
      this.error = null;

      try {
        const config = useRuntimeConfig();

        // Build query string
        const queryParams = new URLSearchParams();

        if (params.query) queryParams.append('query', params.query);

        // Map category_ids to category_id (backend seems to expect singular or maybe handle comma separated)
        // User said: category_id=5. I will send comma separated if multiple, hoping backend handles it, 
        // or just send multiple parameters? URLSearchParams handles multiple with same key, but standard usage varies.
        // Let's try comma separated for now as it's safer for "ids". 
        // User specifically wrote "category_id".
        if (params.categories) queryParams.append('category_id', params.categories);

        // Map price filter to is_free
        if (params.price === 'free') queryParams.append('is_free', '1');
        if (params.price === 'paid') queryParams.append('is_free', '0');

        if (params.sort) queryParams.append('sort', params.sort);
        if (params.page) queryParams.append('page', params.page.toString());

        const queryString = queryParams.toString();

        // Determine endpoint based on whether there is a text query
        // User specified: /api/v1/student/courses/search?query=java
        //                 /api/v1/student/courses?category_id=5
        // I will assume /search requires 'query'.
        let endpoint = '/student/courses';
        if (params.query) {
          endpoint = '/student/courses/search';
        }

        const url = queryString
          ? `${config.public.apiBase}${endpoint}?${queryString}`
          : `${config.public.apiBase}${endpoint}`;

        console.log('[SearchStore] Requesting URL:', url);

        const response = await $fetch<ApiResponse<{
          data: any[];
          current_page: number;
          last_page: number;
          total: number;
          per_page: number;
        }>>(url, {
          method: 'GET',
        });

        console.log('[SearchStore] Raw Response:', response);

        if (response.success && response.data) {
          // Handle API response structure where courses are in 'courses' array
          // and pagination is in 'pagination' object
          const responseData = response.data as any;
          console.log('[SearchStore] Response Data Object:', responseData);

          // API might return data differently for search vs listing? 
          // Assuming standard structure 'courses' array.
          // If response.data IS the array (common in some setups), handle that.
          // But based on previous logs user sent: data: { courses: [...], pagination: {...} }

          this.results = responseData.courses || [];
          console.log('[SearchStore] Mapped Results:', this.results);
          this.query = params.query || '';

          if (responseData.pagination) {
            this.pagination = {
              currentPage: responseData.pagination.current_page,
              totalPages: responseData.pagination.last_page,
              total: responseData.pagination.total,
              perPage: responseData.pagination.per_page,
            };
          } else if (responseData.meta) {
            // Sometimes standard Laravel pagination is in meta
            this.pagination = {
              currentPage: responseData.meta.current_page,
              totalPages: responseData.meta.last_page,
              total: responseData.meta.total,
              perPage: responseData.meta.per_page,
            };
          }

          // Fallback: Populate categories from results if categories list is empty
          if (this.categories.length === 0 && this.results.length > 0) {
            const extractedCategories = new Map();
            this.results.forEach((course: any) => {
              if (course.category && course.category.id) {
                extractedCategories.set(course.category.id, course.category);
              }
            });
            if (extractedCategories.size > 0) {
              this.categories = Array.from(extractedCategories.values());
              console.log('[SearchStore] Extracted categories from results:', this.categories);
            }
          }
        } else {
          console.warn('[SearchStore] Response not successful or no data', response);
        }
      } catch (error: any) {
        console.error('Search error:', error);
        this.error = error.message || 'Search failed';
        throw error;
      } finally {
        this.loading = false;
      }
    },

    /**
     * Fetch all categories
     */
    async fetchCategories() {
      // Skip if already fetched to prevent duplicate API calls
      if (this.categoriesFetched && this.categories.length > 0) {
        return;
      }

      try {
        const config = useRuntimeConfig();
        // Updated endpoint as per user instruction
        const response = await $fetch<ApiResponse<any>>(`${config.public.apiBase}/student/courses/categories`, {
          method: 'GET',
        });

        console.log('[SearchStore] fetchCategories response:', response);

        if (response.success) {
          // Handle both array and object with categories property
          let categoriesData: Category[] = [];
          
          if (Array.isArray(response.data)) {
            categoriesData = response.data;
          } else if (response.data && Array.isArray(response.data.categories)) {
            categoriesData = response.data.categories;
          }
          
          this.categories = categoriesData;
          this.categoriesFetched = true; // Mark as fetched
        } else {
          console.error('[SearchStore] Failed to fetch categories:', response.message);
          // Fallback to empty array to avoid UI errors
          this.categories = [];
        }
      } catch (error: any) {
        console.error('[SearchStore] Fetch categories error:', error);
        this.categories = [];
        // Optional: don't throw here to prevent page crash, just log it
      }
    },

    /**
     * Reset search
     */
    reset() {
      this.query = '';
      this.results = [];
      this.loading = false;
      this.error = null;
    },
  },
});
