import { defineStore } from 'pinia';
import type { ApiResponse } from '~/types/api';

interface PaymentState {
  initiating: boolean;
  executing: boolean;
  retrying: boolean;
  paymentError: string | null;
  currentPaymentId: string | null;
  paymentUrl: string | null;
  pendingPayments: PendingPayment[];
  loadingPending: boolean;
}

interface PendingPayment {
  id: number;
  enrollment_id: number;
  course_id: number;
  course_name: string;
  amount: string;
  status: string;
  created_at: string;
}

export const usePaymentStore = defineStore('payment', {
  state: (): PaymentState => ({
    initiating: false,
    executing: false,
    retrying: false,
    paymentError: null,
    currentPaymentId: null,
    paymentUrl: null,
    pendingPayments: [],
    loadingPending: false,
  }),

  actions: {
    /**
     * Initiate bKash payment for course enrollment
     */
    async initiatePayment(courseId: number, enrollmentId?: number) {
      this.initiating = true;
      this.paymentError = null;

      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<{
          payment_id: string;
          bkash_url: string;
        }>>(`${config.public.apiBase}/payment/initiate`, {
          method: 'POST',
          body: {
            course_id: courseId,
            enrollment_id: enrollmentId,
          },
        });

        if (response.success) {
          this.currentPaymentId = response.data.payment_id;
          this.paymentUrl = response.data.bkash_url;
          
          return {
            success: true,
            paymentId: response.data.payment_id,
            paymentUrl: response.data.bkash_url,
          };
        } else {
          throw new Error(response.message || 'Failed to initiate payment');
        }
      } catch (error: any) {
        this.paymentError = error.message || 'Failed to initiate payment';
        throw error;
      } finally {
        this.initiating = false;
      }
    },

    /**
     * Execute bKash payment after user approval
     */
    async executePayment(paymentID: string) {
      this.executing = true;
      this.paymentError = null;

      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<{
          trxID: string;
          enrollment_id: number;
        }>>(`${config.public.apiBase}/payment/execute`, {
          method: 'POST',
          body: {
            payment_id: paymentID,
          },
        });

        if (response.success) {
          return {
            success: true,
            transactionId: response.data.trxID,
            enrollmentId: response.data.enrollment_id,
            message: response.message,
          };
        } else {
          throw new Error(response.message || 'Payment execution failed');
        }
      } catch (error: any) {
        this.paymentError = error.message || 'Payment execution failed';
        throw error;
      } finally {
        this.executing = false;
      }
    },

    /**
     * Query payment status
     */
    async queryPaymentStatus(paymentID: string) {
      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<{
          transaction_status: string;
        }>>(`${config.public.apiBase}/payment/query/${paymentID}`, {
          method: 'GET',
        });

        if (response.success) {
          return {
            success: true,
            status: response.data.transaction_status,
            data: response.data,
          };
        } else {
          throw new Error(response.message || 'Failed to query payment status');
        }
      } catch (error: any) {
        console.error('Query payment status error:', error);
        throw error;
      }
    },

    /**
     * Search payment by transaction ID
     */
    async searchPayment(trxID: string) {
      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<any>>(`${config.public.apiBase}/payment/search/${trxID}`, {
          method: 'GET',
        });

        if (response.success) {
          return {
            success: true,
            data: response.data,
          };
        } else {
          throw new Error(response.message || 'Failed to search payment');
        }
      } catch (error: any) {
        console.error('Search payment error:', error);
        throw error;
      }
    },

    /**
     * Reset payment state
     */
    reset() {
      this.initiating = false;
      this.executing = false;
      this.retrying = false;
      this.paymentError = null;
      this.currentPaymentId = null;
      this.paymentUrl = null;
    },

    /**
     * Retry failed payment
     */
    async retryPayment(enrollmentId: number) {
      this.retrying = true;
      this.paymentError = null;

      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<{
          payment_id: string;
          bkash_url: string;
        }>>(`${config.public.apiBase}/payment/retry`, {
          method: 'POST',
          body: {
            enrollment_id: enrollmentId,
          },
        });

        if (response.success) {
          this.currentPaymentId = response.data.payment_id;
          this.paymentUrl = response.data.bkash_url;
          
          return {
            success: true,
            paymentId: response.data.payment_id,
            paymentUrl: response.data.bkash_url,
            message: response.message,
          };
        } else {
          throw new Error(response.message || 'Failed to retry payment');
        }
      } catch (error: any) {
        this.paymentError = error?.data?.message || error.message || 'Failed to retry payment';
        throw error;
      } finally {
        this.retrying = false;
      }
    },

    /**
     * Get pending payments
     */
    async fetchPendingPayments() {
      this.loadingPending = true;

      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<PendingPayment[]>>(`${config.public.apiBase}/payment/pending`, {
          method: 'GET',
        });

        if (response.success) {
          this.pendingPayments = response.data || [];
          return this.pendingPayments;
        } else {
          throw new Error(response.message || 'Failed to fetch pending payments');
        }
      } catch (error: any) {
        console.error('Fetch pending payments error:', error);
        this.pendingPayments = [];
        return [];
      } finally {
        this.loadingPending = false;
      }
    },

    /**
     * Get payment status by ID
     */
    async getPaymentStatus(paymentId: number) {
      try {
        const config = useRuntimeConfig();
        const response = await useAuthenticatedFetch<ApiResponse<{
          id: number;
          status: string;
          amount: string;
          transaction_id: string | null;
          enrollment_id: number;
        }>>(`${config.public.apiBase}/payment/status/${paymentId}`, {
          method: 'GET',
        });

        if (response.success) {
          return {
            success: true,
            data: response.data,
          };
        } else {
          throw new Error(response.message || 'Failed to get payment status');
        }
      } catch (error: any) {
        console.error('Get payment status error:', error);
        throw error;
      }
    },
  },
});
