import { defineStore } from 'pinia'

interface Course {
  id: number
  title: string
  slug: string
  thumbnail_path: string
  description: string
  fee_individual: string
  discount_percentage: string
  category: {
    id: number
    name: string
    slug: string
  }
  trainer: {
    id: number
    user_id: number
    trainer_type: string
    name?: string
    user?: {
      id: number
      name: string
    }
  }
  lectures: Lecture[]
}

interface Lecture {
  id: number
  course_id: number
  title: string
  description: string
  topics_covered: string
  video_url: string
  video_thumbnail: string | null
  is_free_preview: boolean
  duration_minutes: number
  sequence_order: number
  status: string
}

interface Enrollment {
  id: number
  student_id: number
  course_id: number
  course: Course
  enrollment_type: string
  amount_paid: string
  discount_applied: string
  enrolled_at: string
  expires_at: string | null
  status: string
  lectures_completed: number
  lectures_total: number
  completion_percentage: string
  last_accessed_at: string | null
  completed_at: string | null
}

interface EnrollmentDetails extends Enrollment {
  lecture_progress: LectureProgress[]
}

interface LectureProgress {
  lecture_id: number
  watched_seconds: number
  is_completed: boolean
  completed_at: string | null
}

interface LectureProgressResponse {
  id?: number
  enrollment_id: number
  lecture_id: number
  is_completed: boolean
  watch_duration_seconds: number
  video_duration_seconds: number
  completion_percentage: number
  watch_count: number
  last_watched_at: string | null
  completed_at: string | null
  course_completion_percentage?: number
  can_download_certificate?: boolean
}

interface LectureProgressPayload {
  enrollment_id: number
  lecture_id: number
  watch_duration_seconds: number
  video_duration_seconds: number
  completion_percentage: number
  is_completed: boolean
}

export const useMyLearningStore = defineStore('myLearning', () => {
  const config = useRuntimeConfig()
  const baseURL = config.public.apiBase
  const authStore = useAuthStore()

  // State
  const enrollments = ref<Enrollment[]>([])
  const enrollmentsLoading = ref(false)
  const enrollmentsError = ref<string | null>(null)
  const enrollmentsLoaded = ref(false)

  const currentEnrollment = ref<EnrollmentDetails | null>(null)
  const currentEnrollmentLoading = ref(false)
  const currentEnrollmentError = ref<string | null>(null)

  // Fetch all enrollments (My Learning)
  const fetchEnrollments = async () => {
    if (!authStore.token) {
      enrollmentsError.value = 'Not authenticated'
      return
    }

    enrollmentsLoading.value = true
    enrollmentsError.value = null

    try {
      const data = await $fetch<{ success: boolean; data: Enrollment[] }>(`${baseURL}/student/my-learning`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      if (data?.success) {
        enrollments.value = data.data || []
        enrollmentsLoaded.value = true
      }
    } catch (err: any) {
      console.error('Enrollments fetch error:', err)
      enrollmentsError.value = err?.data?.message || err?.message || 'Failed to fetch enrollments'
    } finally {
      enrollmentsLoading.value = false
    }
  }

  // Fetch single enrollment details
  const fetchEnrollmentById = async (courseId: string | number) => {
    if (!authStore.token) {
      currentEnrollmentError.value = 'Not authenticated'
      return
    }

    currentEnrollmentLoading.value = true
    currentEnrollmentError.value = null

    try {
      // Fetch student profile to get student_id
      let studentId = null
      try {
        const profileData = await $fetch<{ success: boolean; data: any }>(`${baseURL}/student/profile`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${authStore.token}`
          }
        })
        if (profileData?.success && profileData.data?.student?.id) {
          studentId = profileData.data.student.id
        }
      } catch (error) {
        console.log('Could not fetch student ID from profile')
      }

      // First, try to get the course with enrollment details
      let data
      try {
        data = await $fetch<{ success: boolean; data: any }>(`${baseURL}/student/courses/${courseId}`, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': `Bearer ${authStore.token}`
          }
        })
      } catch (error: any) {
        // If that fails, try alternative endpoint for enrolled courses
        console.log('Primary endpoint failed, trying alternative...')
        data = await $fetch<{ success: boolean; data: any }>(`${baseURL}/courses/${courseId}`, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': `Bearer ${authStore.token}`
          }
        })
      }

      if (data?.success && data.data) {
        const course = data.data.course || data.data
        let enrollment = data.data.enrollment
        
        if (!course) {
          throw new Error('Course not found')
        }

        // If enrollment not provided directly, find it from course.enrollments array
        if (!enrollment && course.enrollments && course.enrollments.length > 0) {
          if (studentId) {
            // Find enrollment matching current student
            enrollment = course.enrollments.find((e: any) => e.student_id === studentId)
            console.log(`Found enrollment for student_id ${studentId}:`, enrollment)
          }
          
          // If still not found, try to find any active enrollment (fallback)
          if (!enrollment) {
            enrollment = course.enrollments.find((e: any) => e.status === 'ACTIVE')
            console.log('Using first active enrollment as fallback:', enrollment)
          }
        }

        // Check if user has an enrollment
        if (!enrollment) {
          throw new Error('You are not enrolled in this course. Please enroll first to access the content.')
        }

        // Build the enrollment object
        currentEnrollment.value = {
          id: enrollment.id,
          student_id: enrollment.student_id,
          course_id: enrollment.course_id || course.id,
          enrollment_type: enrollment.enrollment_type,
          amount_paid: enrollment.amount_paid,
          discount_applied: enrollment.discount_applied || '0.00',
          enrolled_at: enrollment.enrolled_at,
          expires_at: enrollment.expires_at || null,
          status: enrollment.status,
          lectures_completed: enrollment.lectures_completed || 0,
          lectures_total: enrollment.lectures_total || course.lectures?.length || 0,
          completion_percentage: enrollment.completion_percentage || '0.00',
          last_accessed_at: enrollment.last_accessed_at,
          completed_at: enrollment.completed_at,
          lecture_progress: [],
          course: {
            id: course.id,
            title: course.title,
            slug: course.slug,
            description: course.description,
            thumbnail_path: course.thumbnail_path,
            fee_individual: course.fee_individual || '0.00',
            discount_percentage: course.discount_percentage || '0.00',
            category: course.category || { id: 0, name: '', slug: '' },
            trainer: course.trainer || { id: 0, name: '', email: '' },
            lectures: course.lectures || []
          }
        }
      }
    } catch (err: any) {
      console.error('Enrollment details fetch error:', err)
      const errorMessage = err?.data?.message || err?.message || 'Failed to fetch enrollment details'
      currentEnrollmentError.value = errorMessage
      
      // Check if it's a backend SQL error
      if (errorMessage.includes('user_id') || errorMessage.includes('SQLSTATE')) {
        currentEnrollmentError.value = 'Unable to load course. This appears to be a backend configuration issue. Please contact support.'
      }
    } finally {
      currentEnrollmentLoading.value = false
    }
  }

  // Start an enrollment (begin learning)
  const startEnrollment = async (enrollmentId: string | number) => {
    if (!authStore.token) {
      throw new Error('Not authenticated')
    }

    try {
      const data = await $fetch<{ success: boolean; message: string }>(`${baseURL}/student/my-learning/${enrollmentId}/start`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      return data
    } catch (err: any) {
      console.error('Start enrollment error:', err)
      throw new Error(err?.data?.message || err?.message || 'Failed to start enrollment')
    }
  }

  // Update lecture progress (new API)
  const updateLectureProgress = async (payload: LectureProgressPayload): Promise<LectureProgressResponse | null> => {
    console.log('[myLearningStore] updateLectureProgress called:', payload)
    
    if (!authStore.token) {
      console.warn('[myLearningStore] updateLectureProgress: No auth token')
      return null
    }

    try {
      console.log('[myLearningStore] updateLectureProgress: Making API call to', `${baseURL}/student/lecture-progress`)
      const data = await $fetch<{ success: boolean; message: string; data: LectureProgressResponse }>(`${baseURL}/student/lecture-progress`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        },
        body: payload
      })
      
      console.log('[myLearningStore] updateLectureProgress: API response:', data)

      if (data?.success && data.data) {
        // Update local enrollment state if course completion changed
        if (currentEnrollment.value && data.data.course_completion_percentage !== undefined) {
          currentEnrollment.value.completion_percentage = String(data.data.course_completion_percentage)
          
          // Update lectures_completed count if lecture was just completed
          if (data.data.is_completed) {
            // Recalculate completed lectures
            const total = currentEnrollment.value.lectures_total || 1
            currentEnrollment.value.lectures_completed = Math.round((data.data.course_completion_percentage / 100) * total)
          }
        }
        
        return data.data
      }
      return null
    } catch (err: any) {
      console.error('[myLearningStore] updateLectureProgress error:', err)
      return null
    }
  }

  // Get lecture progress for resume (new API)
  const fetchLectureProgress = async (enrollmentId: number, lectureId: number): Promise<LectureProgressResponse | null> => {
    console.log('[myLearningStore] fetchLectureProgress called:', { enrollmentId, lectureId })
    
    if (!authStore.token) {
      console.warn('[myLearningStore] fetchLectureProgress: No auth token')
      return null
    }

    try {
      console.log('[myLearningStore] fetchLectureProgress: Making API call to', `${baseURL}/student/lecture-progress/${enrollmentId}/${lectureId}`)
      const data = await $fetch<{ success: boolean; data: LectureProgressResponse }>(`${baseURL}/student/lecture-progress/${enrollmentId}/${lectureId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })
      
      console.log('[myLearningStore] fetchLectureProgress: API response:', data)

      return data?.data || null
    } catch (err: any) {
      // 404 means no progress yet, which is fine
      if (err?.response?.status !== 404) {
        console.error('[myLearningStore] fetchLectureProgress error:', err)
      } else {
        console.log('[myLearningStore] fetchLectureProgress: No progress found (404)')
      }
      return null
    }
  }

  // Mark lecture as complete
  const markLectureComplete = async (enrollmentId: string | number, lectureId: number) => {
    if (!authStore.token) {
      throw new Error('Not authenticated')
    }

    try {
      const data = await $fetch<{ success: boolean; message: string }>(`${baseURL}/student/my-learning/${enrollmentId}/progress/lecture/${lectureId}/complete`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      // Update local state
      if (currentEnrollment.value) {
        currentEnrollment.value.lectures_completed += 1
        const total = currentEnrollment.value.lectures_total || 1
        currentEnrollment.value.completion_percentage = String((currentEnrollment.value.lectures_completed / total) * 100)
      }

      return data
    } catch (err: any) {
      console.error('Mark complete error:', err)
      throw new Error(err?.data?.message || err?.message || 'Failed to mark lecture as complete')
    }
  }

  // Get lecture progress
  const getLectureProgress = async (enrollmentId: string | number, lectureId: number) => {
    if (!authStore.token) {
      return null
    }

    try {
      const data = await $fetch<{ success: boolean; data: LectureProgress }>(`${baseURL}/student/my-learning/${enrollmentId}/progress/lecture/${lectureId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      return data?.data || null
    } catch (err: any) {
      console.error('Get lecture progress error:', err)
      return null
    }
  }

  // Reset store
  const reset = () => {
    enrollments.value = []
    enrollmentsLoaded.value = false
    enrollmentsError.value = null
    currentEnrollment.value = null
    currentEnrollmentError.value = null
  }

  return {
    // State
    enrollments,
    enrollmentsLoading,
    enrollmentsError,
    enrollmentsLoaded,
    currentEnrollment,
    currentEnrollmentLoading,
    currentEnrollmentError,
    // Actions
    fetchEnrollments,
    fetchEnrollmentById,
    startEnrollment,
    updateLectureProgress,
    fetchLectureProgress,
    markLectureComplete,
    getLectureProgress,
    reset
  }
})
