import { defineStore } from 'pinia'

interface EnrollmentResponse {
  enrollment_id: number
  status: string
  payment_required: boolean
  payment_url?: string
  message: string
}

interface EnrollmentStatus {
  id: number
  course_id: number
  status: string
  payment_status?: string
  enrolled_at?: string
}

export const useEnrollmentStore = defineStore('enrollment', () => {
  const config = useRuntimeConfig()
  const baseURL = config.public.apiBase
  const authStore = useAuthStore()

  // State
  const enrolling = ref(false)
  const enrollmentError = ref<string | null>(null)
  const currentEnrollmentId = ref<number | null>(null)
  const pendingEnrollments = ref<EnrollmentStatus[]>([])

  // Enroll in a course
  const enrollInCourse = async (courseId: string | number) => {
    if (!authStore.token) {
      throw new Error('Please login to enroll')
    }

    enrolling.value = true
    enrollmentError.value = null

    try {
      const data = await $fetch<{ success: boolean; message: string; data: EnrollmentResponse }>(`${baseURL}/student/enroll/${courseId}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      if (data?.success) {
        // Handle inconsistent API response structure
        const responseData = data.data as any;
        const enrollmentId = responseData.enrollment?.id || responseData.enrollment_id;
        // Check for payment URL in different possible locations
        const paymentUrl = responseData.payment?.bkash_url || responseData.payment_url;

        currentEnrollmentId.value = enrollmentId;

        return {
          success: true,
          enrollmentId: enrollmentId,
          status: responseData.enrollment?.status || responseData.status,
          paymentRequired: !!paymentUrl,
          paymentUrl: paymentUrl,
          message: data.message
        }
      }

      throw new Error(data?.message || 'Enrollment failed')
    } catch (err: any) {
      console.error('Enrollment error:', err)
      enrollmentError.value = err?.data?.message || err?.message || 'Failed to enroll'
      throw new Error(enrollmentError.value || 'Failed to enroll')
    } finally {
      enrolling.value = false
    }
  }

  // Get enrollment status
  const getEnrollmentStatus = async (enrollmentId: string | number) => {
    if (!authStore.token) {
      throw new Error('Not authenticated')
    }

    try {
      const data = await $fetch<{ success: boolean; data: EnrollmentStatus }>(`${baseURL}/student/enroll/status/${enrollmentId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      return data?.data || null
    } catch (err: any) {
      console.error('Get enrollment status error:', err)
      throw new Error(err?.data?.message || err?.message || 'Failed to get enrollment status')
    }
  }

  // Cancel pending enrollment
  const cancelEnrollment = async (enrollmentId: string | number) => {
    if (!authStore.token) {
      throw new Error('Not authenticated')
    }

    try {
      const data = await $fetch<{ success: boolean; message: string }>(`${baseURL}/student/enroll/cancel/${enrollmentId}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      return data
    } catch (err: any) {
      console.error('Cancel enrollment error:', err)
      throw new Error(err?.data?.message || err?.message || 'Failed to cancel enrollment')
    }
  }

  // Get user's enrollments
  const fetchEnrollments = async () => {
    if (!authStore.token) {
      return
    }

    try {
      const data = await $fetch<{ success: boolean; data: EnrollmentStatus[] }>(`${baseURL}/student/enrollments`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${authStore.token}`
        }
      })

      pendingEnrollments.value = data?.data || []
    } catch (err: any) {
      console.error('Fetch enrollments error:', err)
    }
  }

  // Check if user is enrolled in a course
  const isEnrolledInCourse = (courseId: number) => {
    return pendingEnrollments.value.some(e => e.course_id === courseId && e.status === 'ACTIVE')
  }

  // Get enrollment for a course
  const getEnrollmentForCourse = (courseId: number) => {
    return pendingEnrollments.value.find(e => e.course_id === courseId)
  }

  // Reset store
  const reset = () => {
    enrolling.value = false
    enrollmentError.value = null
    currentEnrollmentId.value = null
    pendingEnrollments.value = []
  }

  return {
    // State
    enrolling,
    enrollmentError,
    currentEnrollmentId,
    pendingEnrollments,
    // Actions
    enrollInCourse,
    getEnrollmentStatus,
    cancelEnrollment,
    fetchEnrollments,
    isEnrolledInCourse,
    getEnrollmentForCourse,
    reset
  }
})
