import { defineStore } from 'pinia'

interface DashboardStats {
  total_enrolled: number
  completed_courses: number
  in_progress: number
  pending_payment: number
  certificates_earned: number
  total_spent: number
}

interface EnrolledCourse {
  enrollment_id: number
  course_id: number
  course_name: string
  course_slug: string
  category: string
  trainer: string
  thumbnail: string
  status: string
  payment_status?: string
  payment_pending?: boolean
  can_retry_payment?: boolean
  completion_percentage: number
  lectures_completed: number
  lectures_total: number
  enrolled_at: string
  last_accessed_at: string | null
  can_download_certificate: boolean
}

interface AvailableCourse {
  course_id: number
  course_name: string
  course_slug: string
  description: string
  thumbnail: string
  category: string
  trainer_name: string
  trainer_type: string
  fee_individual: number
  fee_institute: number
  discount_percentage: number
  discounted_individual_fee: number
  discounted_institute_fee: number
  total_lectures: number
  total_duration_minutes: number
  enrolled_students: number
}

interface DashboardData {
  stats: DashboardStats
  enrolled_courses: EnrolledCourse[]
  available_courses: AvailableCourse[]
  has_enrolled_courses: boolean
}

export const useDashboardStore = defineStore('dashboard', () => {
  const config = useRuntimeConfig()
  const baseURL = config.public.apiBase
  const authStore = useAuthStore()

  // State
  const stats = ref<DashboardStats>({
    total_enrolled: 0,
    completed_courses: 0,
    in_progress: 0,
    pending_payment: 0,
    certificates_earned: 0,
    total_spent: 0
  })
  const enrolledCourses = ref<EnrolledCourse[]>([])
  const availableCourses = ref<AvailableCourse[]>([])
  const hasEnrolledCourses = ref(false)
  const loading = ref(false)
  const error = ref<string | null>(null)
  const loaded = ref(false)

  // Fetch dashboard data
  const fetchDashboard = async () => {
    if (!authStore.token) {
      error.value = 'Not authenticated'
      return
    }

    loading.value = true
    error.value = null

    try {
      const data = await useAuthenticatedFetch<{ success: boolean; data: DashboardData }>(`${baseURL}/student/dashboard`, {
        method: 'GET'
      })

      if (data?.success) {
        stats.value = data.data.stats || {
          total_enrolled: 0,
          completed_courses: 0,
          in_progress: 0,
          pending_payment: 0,
          certificates_earned: 0,
          total_spent: 0
        }
        enrolledCourses.value = data.data.enrolled_courses || []
        availableCourses.value = data.data.available_courses || []
        hasEnrolledCourses.value = data.data.has_enrolled_courses
        loaded.value = true
      }
    } catch (err: any) {
      console.error('Dashboard fetch error:', err)
      error.value = err?.data?.message || err?.message || 'Failed to fetch dashboard'
    } finally {
      loading.value = false
    }
  }

  // Reset store
  const reset = () => {
    stats.value = {
      total_enrolled: 0,
      completed_courses: 0,
      in_progress: 0,
      pending_payment: 0,
      certificates_earned: 0,
      total_spent: 0
    }
    enrolledCourses.value = []
    availableCourses.value = []
    hasEnrolledCourses.value = false
    loaded.value = false
    error.value = null
  }

  return {
    stats,
    enrolledCourses,
    availableCourses,
    hasEnrolledCourses,
    loading,
    error,
    loaded,
    fetchDashboard,
    reset
  }
})
