import { defineStore } from 'pinia';
import type { ApiResponse } from '~/types/api';

interface Certificate {
  id: number;
  enrollment_id: number;
  course_id: number;
  course_title: string;
  certificate_number: string;
  issued_date: string;
  completion_date: string;
  certificate_url?: string;
  trainer_name?: string;
  trainer_signature?: string;
}

interface CertificateState {
  certificates: Certificate[];
  currentCertificate: Certificate | null;
  loading: boolean;
  error: string | null;
}

export const useCertificatesStore = defineStore('certificates', {
  state: (): CertificateState => ({
    certificates: [],
    currentCertificate: null,
    loading: false,
    error: null,
  }),

  getters: {
    /**
     * Get certificates count
     */
    certificatesCount: (state) => state.certificates.length,

    /**
     * Get certificates by year
     */
    certificatesByYear: (state) => {
      const groupedByYear: Record<number, Certificate[]> = {};
      
      state.certificates.forEach(cert => {
        const year = new Date(cert.issued_date).getFullYear();
        if (!groupedByYear[year]) {
          groupedByYear[year] = [];
        }
        groupedByYear[year].push(cert);
      });

      return groupedByYear;
    },
  },

  actions: {
    /**
     * Fetch all certificates for the student
     */
    async fetchCertificates() {
      this.loading = true;
      this.error = null;

      try {
        const config = useRuntimeConfig();
        const token = useCookie('access_token');

        const response = await $fetch<ApiResponse<{
        certificates: Certificate[];
      }>>(`${config.public.apiBase}/student/certificates`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${token.value}`,
          },
        });

        if (response.success) {
          this.certificates = response.data.certificates || [];
        } else {
          throw new Error(response.message || 'Failed to fetch certificates');
        }
      } catch (error: any) {
        console.error('Fetch certificates error:', error);
        this.error = error.message || 'Failed to fetch certificates';
        throw error;
      } finally {
        this.loading = false;
      }
    },

    /**
     * Fetch certificate by ID
     */
    async fetchCertificateById(certificateId: number) {
      this.loading = true;
      this.error = null;

      try {
        const config = useRuntimeConfig();
        const token = useCookie('access_token');

        const response = await $fetch<ApiResponse<Certificate>>(`${config.public.apiBase}/student/certificates/${certificateId}`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${token.value}`,
          },
        });

        if (response.success) {
          this.currentCertificate = response.data;
          return response.data;
        } else {
          throw new Error(response.message || 'Failed to fetch certificate');
        }
      } catch (error: any) {
        console.error('Fetch certificate error:', error);
        this.error = error.message || 'Failed to fetch certificate';
        throw error;
      } finally {
        this.loading = false;
      }
    },

    /**
     * Download certificate
     */
    async downloadCertificate(certificateId: number) {
      try {
        const config = useRuntimeConfig();
        const token = useCookie('access_token');

        const response = await $fetch<ApiResponse<{
        download_url: string;
      }>>(`${config.public.apiBase}/student/certificates/${certificateId}/download`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${token.value}`,
          },
        });

        if (response.success && response.data.download_url) {
          // Open download URL in new tab
          window.open(response.data.download_url, '_blank');
          return true;
        } else {
          throw new Error(response.message || 'Failed to download certificate');
        }
      } catch (error: any) {
        console.error('Download certificate error:', error);
        throw error;
      }
    },

    /**
     * Verify certificate by certificate number
     */
    async verifyCertificate(certificateNumber: string) {
      try {
        const config = useRuntimeConfig();

        const response = await $fetch<ApiResponse<any>>(`${config.public.apiBase}/certificates/verify/${certificateNumber}`, {
          method: 'GET',
        });

        if (response.success) {
          return response.data;
        } else {
          throw new Error(response.message || 'Certificate not found');
        }
      } catch (error: any) {
        console.error('Verify certificate error:', error);
        throw error;
      }
    },

    /**
     * Reset certificates state
     */
    reset() {
      this.certificates = [];
      this.currentCertificate = null;
      this.loading = false;
      this.error = null;
    },
  },
});
