import { defineStore } from 'pinia'

interface User {
  id: number
  name: string
  email: string
  mobile: string
  role: string
  is_active?: boolean
  email_verified?: boolean
  created_at?: string
  updated_at?: string
}

interface AuthData {
  user: User
  access_token: string
  token_type: string
  expires_in: number
  login_type?: string
}

interface RegisterPayload {
  name: string
  email: string
  mobile: string
  password: string
  role: string
  terms: boolean
  password_confirmation: string
}

interface LoginPayload {
  login: string
  password: string
  remember: boolean
}

export const useAuthStore = defineStore('auth', () => {
  const config = useRuntimeConfig()
  const baseURL = config.public.apiBase

  // State
  const user = ref<User | null>(null)
  const token = ref<string | null>(null)
  const isAuthenticated = computed(() => !!token.value)

  // Initialize from cookie - using access_token for consistency
  const tokenCookie = useCookie('access_token', {
    maxAge: 60 * 60 * 24 * 7, // 7 days
    sameSite: 'lax',
    secure: process.env.NODE_ENV === 'production'
  })

  // Initialize token from cookie on store creation
  if (tokenCookie.value) {
    token.value = tokenCookie.value as string
  }

  // Register
  const register = async (payload: RegisterPayload) => {
    try {
      const { data, error } = await useFetch<{ success: boolean; message: string; data: AuthData }>(`${baseURL}/auth/register`, {
        method: 'POST',
        body: payload,
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        }
      })

      if (error.value) {
        const err = error.value.data as any
        throw new Error(err?.message || 'Registration failed')
      }

      if (data.value?.success) {
        user.value = data.value.data.user
        token.value = data.value.data.access_token
        tokenCookie.value = data.value.data.access_token
        return { success: true, message: data.value.message }
      }

      throw new Error('Registration failed')
    } catch (err: any) {
      throw err
    }
  }

  // Login
  const login = async (payload: LoginPayload) => {
    try {
      const { data, error } = await useFetch<{ success: boolean; message: string; data: AuthData }>(`${baseURL}/auth/login`, {
        method: 'POST',
        body: payload,
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        }
      })

      if (error.value) {
        const err = error.value.data as any
        throw new Error(err?.message || 'Login failed')
      }

      if (data.value?.success) {
        user.value = data.value.data.user
        token.value = data.value.data.access_token
        tokenCookie.value = data.value.data.access_token
        return { success: true, message: data.value.message }
      }

      throw new Error('Login failed')
    } catch (err: any) {
      throw err
    }
  }

  // Fetch user profile
  const fetchUser = async () => {
    if (!token.value) {
      throw new Error('No token available')
    }

    try {
      const data = await useAuthenticatedFetch<{ success: boolean; message: string; data: User }>(`${baseURL}/auth/me`, {
        method: 'GET'
      })

      if (data?.success) {
        user.value = data.data
        return data.data
      }

      throw new Error('Failed to fetch user')
    } catch (err) {
      console.error('Error fetching user:', err)
      // Only clear if it's NOT a 401 that we failed to refresh (which calls logout inside useAuthenticatedFetch)
      // Actually useAuthenticatedFetch handles logout given 401 refresh failure.
      // But we might want to ensure state is cleared if other errors occur? 
      // Existing code cleared state on error. Let's keep it safe.
      if (!user.value) { 
         // If user is null, it means we probably failed critical auth, so verify.
         // But if useAuthenticatedFetch throws, it means either non-401 error OR 401 refresh failed (and logout called).
      }
      throw err
    }
  }

  // Logout
  const logout = async () => {
    try {
      if (token.value) {
        await $fetch(`${baseURL}/auth/logout`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': `Bearer ${token.value}`
          }
        })
      }
    } catch (err) {
      console.error('Logout error:', err)
    } finally {
      user.value = null
      token.value = null
      tokenCookie.value = null
      
      // Use navigateTo instead of direct navigation
      if (process.client) {
        navigateTo('/login')
      }
    }
  }

  // Refresh token
  const refreshToken = async () => {
    if (!token.value) return

    try {
      const { data, error } = await useFetch<{ success: boolean; message: string; data: { access_token: string; token_type: string; expires_in: number } }>(`${baseURL}/auth/refresh`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': `Bearer ${token.value}`
        }
      })

      if (error.value) {
        throw new Error('Token refresh failed')
      }

      if (data.value?.success) {
        token.value = data.value.data.access_token
        tokenCookie.value = data.value.data.access_token
        return { success: true }
      }

      throw new Error('Token refresh failed')
    } catch (err) {
      console.error('Token refresh error:', err)
      await logout()
    }
  }

  return {
    user,
    token,
    isAuthenticated,
    register,
    login,
    fetchUser,
    logout,
    refreshToken
  }
})