/**
 * Video Progress Tracking Composable
 * 
 * Handles:
 * - YouTube iframe API integration
 * - Auto-save progress every 10-15 seconds
 * - Resume from last watched position
 * - Auto-mark complete at 80%
 */

interface VideoProgressOptions {
  enrollmentId: number
  lectureId: number
  videoDuration?: number
  onComplete?: () => void
  onProgressUpdate?: (progress: number) => void
}

interface ProgressState {
  currentTime: number
  duration: number
  percentage: number
  isCompleted: boolean
  canDownloadCertificate: boolean
  courseCompletionPercentage: number
}

export const useVideoProgress = () => {
  const myLearningStore = useMyLearningStore()
  
  // State
  const isTracking = ref(false)
  const lastSavedTime = ref(0)
  const progressState = ref<ProgressState>({
    currentTime: 0,
    duration: 0,
    percentage: 0,
    isCompleted: false,
    canDownloadCertificate: false,
    courseCompletionPercentage: 0
  })
  
  let saveInterval: ReturnType<typeof setInterval> | null = null
  let currentOptions: VideoProgressOptions | null = null
  
  // Fetch initial progress for resume
  const fetchInitialProgress = async (enrollmentId: number, lectureId: number): Promise<number> => {
    console.log('[useVideoProgress] fetchInitialProgress called:', { enrollmentId, lectureId })
    try {
      const progress = await myLearningStore.fetchLectureProgress(enrollmentId, lectureId)
      console.log('[useVideoProgress] fetchLectureProgress result:', progress)
      
      if (progress) {
        progressState.value.currentTime = progress.watch_duration_seconds || 0
        progressState.value.duration = progress.video_duration_seconds || 0
        progressState.value.percentage = progress.completion_percentage || 0
        progressState.value.isCompleted = progress.is_completed || false
        progressState.value.canDownloadCertificate = progress.can_download_certificate || false
        progressState.value.courseCompletionPercentage = progress.course_completion_percentage || 0
        
        return progress.watch_duration_seconds || 0
      }
      
      return 0
    } catch (error) {
      console.error('[useVideoProgress] Failed to fetch initial progress:', error)
      return 0
    }
  }
  
  // Save progress to API
  const saveProgress = async (
    currentTime: number,
    duration: number,
    forceComplete: boolean = false
  ) => {
    if (!currentOptions) {
      console.warn('[useVideoProgress] saveProgress: No currentOptions set')
      return
    }
    
    const { enrollmentId, lectureId, onComplete, onProgressUpdate } = currentOptions
    
    // Don't save if no significant change (less than 5 seconds difference)
    if (Math.abs(currentTime - lastSavedTime.value) < 5 && !forceComplete) {
      console.log('[useVideoProgress] saveProgress: Skipping, no significant change')
      return
    }
    
    const percentage = duration > 0 ? Math.round((currentTime / duration) * 100) : 0
    const isCompleted = forceComplete || percentage >= 80
    
    const payload = {
      enrollment_id: enrollmentId,
      lecture_id: lectureId,
      watch_duration_seconds: Math.round(currentTime),
      video_duration_seconds: Math.round(duration),
      completion_percentage: Math.min(percentage, 100),
      is_completed: isCompleted
    }
    
    console.log('[useVideoProgress] saveProgress: Calling API with:', payload)
    
    try {
      const result = await myLearningStore.updateLectureProgress(payload)
      console.log('[useVideoProgress] saveProgress: API result:', result)
      
      if (result) {
        lastSavedTime.value = currentTime
        
        // Check if this is a NEW completion (was not completed before, now is)
        const wasCompleted = progressState.value.isCompleted
        const nowCompleted = result.is_completed
        
        progressState.value.currentTime = currentTime
        progressState.value.duration = duration
        progressState.value.percentage = percentage
        progressState.value.isCompleted = result.is_completed
        progressState.value.canDownloadCertificate = result.can_download_certificate || false
        progressState.value.courseCompletionPercentage = result.course_completion_percentage || 0
        
        // Callback for progress update
        if (onProgressUpdate) {
          onProgressUpdate(result.course_completion_percentage || percentage)
        }
        
        // Callback when lecture is completed - only call once (not for already-completed lectures)
        if (nowCompleted && !wasCompleted && onComplete) {
          onComplete()
        }
      }
    } catch (error) {
      console.error('[useVideoProgress] Failed to save progress:', error)
    }
  }
  
  // Start tracking with periodic saves
  const startTracking = (options: VideoProgressOptions) => {
    console.log('[useVideoProgress] startTracking called:', options)
    currentOptions = options
    isTracking.value = true
    
    // Save every 10 seconds
    saveInterval = setInterval(() => {
      console.log('[useVideoProgress] Interval tick:', {
        currentTime: progressState.value.currentTime,
        duration: progressState.value.duration
      })
      if (progressState.value.currentTime > 0 && progressState.value.duration > 0) {
        saveProgress(progressState.value.currentTime, progressState.value.duration)
      }
    }, 10000) // 10 seconds
  }
  
  // Stop tracking and save final position
  const stopTracking = async () => {
    isTracking.value = false
    
    if (saveInterval) {
      clearInterval(saveInterval)
      saveInterval = null
    }
    
    // Save final position
    if (progressState.value.currentTime > 0 && progressState.value.duration > 0) {
      await saveProgress(progressState.value.currentTime, progressState.value.duration)
    }
    
    currentOptions = null
  }
  
  // Update current time (called from video player)
  const updateCurrentTime = (time: number, duration?: number) => {
    progressState.value.currentTime = time
    if (duration) {
      progressState.value.duration = duration
    }
  }
  
  // Mark as complete manually
  const markComplete = async () => {
    if (!currentOptions) return
    
    await saveProgress(
      progressState.value.duration || progressState.value.currentTime,
      progressState.value.duration,
      true // Force complete
    )
  }
  
  // Cleanup on unmount
  onUnmounted(() => {
    if (saveInterval) {
      clearInterval(saveInterval)
    }
  })
  
  return {
    // State
    isTracking: readonly(isTracking),
    progressState: readonly(progressState),
    
    // Methods
    fetchInitialProgress,
    saveProgress,
    startTracking,
    stopTracking,
    updateCurrentTime,
    markComplete
  }
}
