
export const useAuthenticatedFetch = async <T>(url: string, options: any = {}) => {
  const authStore = useAuthStore()
  const config = useRuntimeConfig()
  
  // Helper to get fresh token
  const getToken = () => authStore.token

  // Merge headers
  const headers = {
    ...options.headers,
    'Authorization': `Bearer ${getToken()}`,
    'Accept': 'application/json',
  }

  try {
    return await $fetch<T>(url, {
      ...options,
      headers
    })
  } catch (error: any) {
    // If we caught a 401 and refresh succeeded, we want to retry.
    // However, the simple $fetch wrapper above isn't enough for clean retry logic because $fetch doesn't support "retry on error interceptor" natively in a simple way without a wrapper loop.
    
    // Let's implement the retry loop logic manually.
    if (error?.response?.status === 401) {
      try {
        console.log('[useAuthenticatedFetch] 401 detected, attempting refresh...')
        const refreshResult = await authStore.refreshToken()
        
        if (refreshResult?.success) {
           console.log('[useAuthenticatedFetch] Refresh successful, retrying request...')
           // Retry with new token
           const newToken = getToken()
           const newHeaders = {
             ...options.headers,
             'Authorization': `Bearer ${newToken}`,
             'Accept': 'application/json',
           }
           return await $fetch<T>(url, {
             ...options,
             headers: newHeaders
           })
        }
      } catch (refreshError) {
        console.error('[useAuthenticatedFetch] Refresh failed:', refreshError)
        await authStore.logout()
        throw refreshError
      }
    }
    
    throw error
  }
}
